#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTest.VAL;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.FrmBloodProductEdit;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_BloodProduct : BaseTest 
	{
		private BOL.BloodProduct _testBloodProduct;
		private DataTable _dtBloodProduct;
        private bool _aborhConfirmationRequired = false;

		[SetUp]
		protected void SetUp() 
		{				
			this.BaseSetUp();

			if (this.RefreshData)
			{
                string strSQL = " SELECT TOP 1 H.HcpcsAssignmentGuid, B.BloodProductGuid, B.ProductName, B.ProductTypeCode, B.ProductCode, B.ProductVolume, B.ProductCodeIndicator AS BarcodeType, U.AboRhConfirmationRequired" +
					            " FROM BloodProduct B " +
								" JOIN HcpcsAssignment H ON B.BloodProductGuid = H.BloodProductGuid " +
                                " JOIN vw_BloodShipPatientAllCurrent U ON U.BloodProductGuid = B.BloodProductGuid AND U.RecordStatusCode = 'A' AND U.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'" + 
								" AND U.UnitStatusCode = 'A' " + 
								" WHERE B.ProductCode LIKE 'E%'" +
								" AND B.BloodProductGuid IN (SELECT DISTINCT BloodProductGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "')";

				this._dtBloodProduct = UnitTestUtil.RunSQLGetDataTable(strSQL, false);
				if (this._dtBloodProduct.Rows.Count == 0)
				{
					DataCreator.CreateBloodUnit(false);
					this._dtBloodProduct = UnitTestUtil.RunSQLGetDataTable(strSQL, true);
				}
				this.RefreshData = false;
			}

            this._aborhConfirmationRequired = System.Convert.ToBoolean(this._dtBloodProduct.Rows[0][TABLES.ProductType.AboRhConfirmationRequired]);
			this._testBloodProduct = new BOL.BloodProduct((System.Guid)this._dtBloodProduct.Rows[0][TABLES.BloodProduct.BloodProductGuid]);
			this._testBloodProduct.SetInitialBrokenRules(BROKEN.FormClassName);
		}

		[Test]
		public void Ctor_Blank_Pass()
		{
			BOL.BloodProduct testObj = new BOL.BloodProduct();

			Assert.IsTrue(testObj.BloodProductGuid == System.Guid.Empty);
			Assert.IsTrue(testObj.ProductCode == string.Empty);
			Assert.IsTrue(testObj.ProductShortName == string.Empty);
			Assert.IsTrue(testObj.ProductName == string.Empty);
			Assert.IsTrue(testObj.ProductVolume == 0);
			Assert.IsTrue(testObj.MaximumStorageDays == 0);
			Assert.IsTrue(testObj.MaximumStorageHours == 0);
			Assert.IsTrue(testObj.ProductActiveIndicator == false);
			Assert.IsTrue(testObj.BarcodeType == Common.BarcodeType.None);
            Assert.IsTrue(testObj.ProductType.AboRhConfirmationRequired == false);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Blank_Fail()
		{
		}


		[Test]
		public void Ctor_Guid_Pass()
		{
			DataRow drData = this._dtBloodProduct.Rows[0];
			BOL.BloodProduct testObj = new BloodProduct(this._testBloodProduct.BloodProductGuid);

			Assert.IsTrue(testObj.BloodProductGuid == (System.Guid)drData[TABLES.BloodProduct.BloodProductGuid]);
			Assert.IsTrue(testObj.ProductCode == drData[TABLES.BloodProduct.ProductCode].ToString());
			Assert.IsTrue(testObj.ProductName == drData[TABLES.BloodProduct.ProductName].ToString());
			Assert.IsTrue(testObj.ProductType.ProductTypeCode == drData[TABLES.BloodProduct.ProductTypeCode].ToString());
			Assert.IsTrue(testObj.ProductVolume == (short)drData[TABLES.BloodProduct.ProductVolume]);
            Assert.AreEqual(testObj.ProductType.AboRhConfirmationRequired, _aborhConfirmationRequired);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Guid_Fail()
		{
		}

		[Test]
		public void Ctor_Datarow_Pass()
		{
			DataRow drData = this._dtBloodProduct.Rows[0];
			BOL.BloodProduct testObj = new BOL.BloodProduct(drData);
			
			Assert.IsTrue(testObj.BloodProductGuid == (System.Guid)drData[TABLES.BloodProduct.BloodProductGuid]);
			Assert.IsTrue(testObj.ProductCode == drData[TABLES.BloodProduct.ProductCode].ToString());
			Assert.IsTrue(testObj.ProductName == drData[TABLES.BloodProduct.ProductName].ToString());
			Assert.IsTrue(testObj.ProductType.ProductTypeCode == drData[TABLES.BloodProduct.ProductTypeCode].ToString());
			Assert.IsTrue(testObj.ProductVolume == (short)drData[TABLES.BloodProduct.ProductVolume]);
            Assert.AreEqual(testObj.ProductType.AboRhConfirmationRequired, _aborhConfirmationRequired);
			Assert.IsTrue(testObj.IsDirty == false);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Datarow_Fail()
		{
		}

		[Test]
		public void BloodProductGUID_Pass()
		{
			System.Guid tmpValue = System.Guid.NewGuid();
			this._testBloodProduct.BloodProductGuid = tmpValue;
			Assert.IsTrue(this._testBloodProduct.BloodProductGuid == tmpValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodProductGUID_Fail()
		{
		}

		[Test]
		public void ProductAttributeForm_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();
			this._testBloodProduct.ProductAttributeForm = tmpValue;
			Assert.IsTrue(this._testBloodProduct.ProductAttributeForm == tmpValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ProductAttributeForm_Fail()
		{
		}

		[Test]
		public void IsFrozen_Pass()
		{
			BOL.BloodProduct objTest = new BOL.BloodProduct();

			objTest.ProductType.ProductTypeCode = "E004";
			Assert.IsTrue(objTest.IsFrozen, "E004");

			objTest.ProductType.ProductTypeCode = "E005";
			Assert.IsTrue(objTest.IsFrozen, "E005");
			
			objTest.ProductType.ProductTypeCode = "E010";
			Assert.IsTrue(objTest.IsFrozen, "E010");
			
			objTest.ProductType.ProductTypeCode = "E012";
			Assert.IsTrue(objTest.IsFrozen, "E012");
			
			objTest.ProductType.ProductTypeCode = "E025";
			Assert.IsTrue(objTest.IsFrozen, "E025");
			
			objTest.ProductType.ProductTypeCode = "E028";
			Assert.IsTrue(objTest.IsFrozen, "E028");
			
			objTest.ProductType.ProductTypeCode = "E030";
			Assert.IsTrue(objTest.IsFrozen, "E030");
			
			objTest.ProductType.ProductTypeCode = "E032";
			Assert.IsTrue(objTest.IsFrozen, "E032");
			
			objTest.ProductType.ProductTypeCode = "E048";
			Assert.IsTrue(objTest.IsFrozen, "E048");
			
			objTest.ProductType.ProductTypeCode = "E050";
			Assert.IsTrue(objTest.IsFrozen, "E050");
			
			objTest.ProductType.ProductTypeCode = "E053";
			Assert.IsTrue(objTest.IsFrozen, "E053");			
			
			objTest.ProductType.ProductTypeCode = "E123";
			Assert.IsTrue(objTest.IsFrozen == false);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IsFrozen_Fail()
		{
		}

		[Test]
		public void ProductCode_Pass()
		{
			BOL.BloodProduct objTest = new BOL.BloodProduct();
			objTest.ProductCode = "04020";
			Assert.IsTrue(objTest.ProductCode == "04020");
			Assert.IsTrue(objTest.BarcodeType == Common.BarcodeType.Codabar);

			objTest.ProductCode = "=<E1234V56";
			
			Assert.IsTrue(objTest.DonationType.DonationTypeCode == 'V', "DonationTypeCode");
			Assert.IsTrue(objTest.ProductCode == "E1234", "ProductCode");
			Assert.IsTrue(objTest.BarcodeType == Common.BarcodeType.ISBT128, "BarcodeType");
			Assert.IsTrue(objTest.UnitDivisionId == "56", "UnitDivisionId");
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void ProductCode_Fail1()
		{
			BOL.BloodProduct objTest = new BOL.BloodProduct();
			objTest.ProductCode = "1";
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void ProductCode_Fail2()
		{
			BOL.BloodProduct objTest = new BOL.BloodProduct();
			objTest.ProductCode = "=<E1234B56";
		}

		[Test]
		public void ProductShortName_Pass()
		{
			string tmpValue = System.DateTime.Now.Minute.ToString();
			this._testBloodProduct.ProductShortName = tmpValue;

			Assert.IsTrue(this._testBloodProduct.ProductShortName == tmpValue);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void ProductShortName_Fail()
		{
			this._testBloodProduct.ProductShortName = string.Empty.PadRight(51,'X');
		}

		[Test]
		public void ProductName_Pass()
		{
			string tmpValue = System.DateTime.Now.ToString();
			this._testBloodProduct.ProductName = tmpValue;
			Assert.IsTrue(this._testBloodProduct.ProductName == tmpValue);
		}

		[Test]
		[ExpectedException(typeof(BOL.BusinessObjectException))]
		public void ProductName_Fail()
		{
			this._testBloodProduct.ProductName = string.Empty.PadRight(176,'X');
		}

		[Test]
		public void ProductType_Pass()
		{
			BOL.ProductType objTest = new BOL.ProductType();
			this._testBloodProduct.ProductType = objTest;
			Assert.IsTrue(this._testBloodProduct.ProductType == objTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ProductType_Fail()
		{
		}

		[Test]
		public void ProductVolume_Pass()
		{
			int tmpValue = new Random().Next(2000);
			this._testBloodProduct.ProductVolume = (short)tmpValue;
			Assert.IsTrue(this._testBloodProduct.ProductVolume == (short)tmpValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void ProductVolume_Fail1()
		{
			this._testBloodProduct.ProductVolume = short.MinValue;
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void ProductVolume_Fail2()
		{
			this._testBloodProduct.ProductVolume = 2001;
		}

		[Test]
		public void MaximumStorageDays_Pass()
		{
			int tmpValue = new Random().Next(5000);
			this._testBloodProduct.MaximumStorageDays = tmpValue;
			Assert.IsTrue(this._testBloodProduct.MaximumStorageDays == tmpValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void MaximumStorageDays_Fail()
		{
			this._testBloodProduct.MaximumStorageDays = int.MinValue;
		}

		[Test]
		public void MaximumStorageHours_Pass()
		{
			int tmpValue = new Random().Next(23);
			this._testBloodProduct.MaximumStorageHours = tmpValue;
			Assert.IsTrue(this._testBloodProduct.MaximumStorageHours == tmpValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void MaximumStorageHours_Fail1()
		{
			this._testBloodProduct.MaximumStorageHours = int.MinValue;
		}

		[Test]
		public void IsBloodProductOnActiveUnits_Pass()
		{
			DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM ProductShipper PS WHERE DivisionCode = '" + this.TestDivisionCode + "' AND RecordStatusCode = 'A' AND BloodProductGuid IN (SELECT DISTINCT BloodProductGuid FROM vw_BloodShipAllCurrent V WHERE PS.LocalSupplierGuid = V.LocalSupplierGuid AND V.UnitStatusCode IN ('A','S','L'))");

			System.Guid lsGuid = (System.Guid)dtTest.Rows[0][TABLES.ProductShipper.LocalSupplierGuid];
			System.Guid bpGuid = (System.Guid)dtTest.Rows[0][TABLES.ProductShipper.BloodProductGuid];

			bool bretVal = BOL.BloodProduct.IsBloodProductOnActiveUnits(bpGuid, lsGuid);
			Assert.IsTrue(bretVal);
		}

		[Test]
		public void IsBloodProductOnActiveUnits_Fail()
		{
			System.Guid bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM BloodProduct WHERE BloodProductGuid NOT IN (SELECT DISTINCT BloodProductGuid FROM ProductShipper WHERE DivisionCode = '" + this.TestDivisionCode + "')");

			System.Guid lsGuid = DataCreator.GetRecordGuid("SELECT TOP 1 LocalSupplierGuid FROM LocalSupplier WHERE DivisionCode = '" + this.TestDivisionCode + "'");

			BOL.ProductShipper newPS = new BOL.ProductShipper(System.Guid.NewGuid(), bpGuid);
			newPS.ReturnCreditPercentage = (decimal)99;
			newPS.ProductCost = (decimal)50;
			newPS.LocalSupplierGuid = lsGuid;

			bool addOK = newPS.Insert(Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, addOK, "New ProductShipper");

			bool retVal = BOL.BloodProduct.IsBloodProductOnActiveUnits(bpGuid, lsGuid);
			Assert.AreEqual(false, retVal, "Active check");
		}


		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void MaximumStorageHours_Fail2()
		{
			this._testBloodProduct.MaximumStorageHours = 24;
		}

		[Test]
		public void ProductActiveIndicator_Pass()
		{
			bool tstValue = this._testBloodProduct.ProductActiveIndicator;
			Assert.AreEqual(tstValue, this._testBloodProduct.ProductActiveIndicator);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void ProductActiveIndicator_Fail()
		{
		}

		[Test]
		public void BarcodeType_Pass()
		{
			Common.BarcodeType tmpValue = Common.BarcodeType.ISBT128;
			this._testBloodProduct.BarcodeType = tmpValue;
			Assert.IsTrue(this._testBloodProduct.BarcodeType == tmpValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BarcodeType_Fail()
		{
		}

		[Test]
		public void HpcpsCode_Pass()
		{
			string tmpValue = new Random().Next(99999).ToString();
			this._testBloodProduct.HcpcsCode = tmpValue;
			Assert.IsTrue(this._testBloodProduct.HcpcsCode == tmpValue);
		}

		[Test]
		[ExpectedException(typeof(ArgumentException))]
		public void HpcpsCode_Fail()
		{
			this._testBloodProduct.HcpcsCode = string.Empty.PadRight(6,'X');
		}

		[Test]
		public void ProductSuppliers_Pass()
		{
			BOL.ProductShipperCollection objTest = new BOL.ProductShipperCollection(this._testBloodProduct.BloodProductGuid);

			this._testBloodProduct.ProductShippers = objTest;
			Assert.IsTrue(this._testBloodProduct.ProductShippers == objTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ProductSuppliers_Fail()
		{
		}

		[Test]
		public void DonationType_Pass()
		{
			BOL.DonationType objTest = new BOL.DonationType('1');
			this._testBloodProduct.DonationType = objTest;
			Assert.IsTrue(this._testBloodProduct.DonationType == objTest);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DonationType_Fail()
		{
		}

		[Test]
		public void UnitDivisionID_Pass()
		{
			string tmpValue = new Random(10).Next(99).ToString();
			this._testBloodProduct.UnitDivisionId = tmpValue;
			Assert.IsTrue(this._testBloodProduct.UnitDivisionId == tmpValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void UnitDivisionID_Fail()
		{
			this._testBloodProduct.UnitDivisionId = string.Empty.PadRight(3, 'X');
		}

		[Test]
		public void IsProductSupplierActive_Pass()
		{
			Assert.IsTrue(this._testBloodProduct.IsProductShipperActive(System.Guid.NewGuid()) == false);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void IsProductSupplierActive_Fail()
		{
		}

		[Test]
		public void BloodProductSearch_Pass()
		{
			DataTable dt = BOL.BloodProduct.BloodProductSearch(this._testBloodProduct.ProductName, this._testBloodProduct.ProductType.ProductTypeCode);
			Assert.IsTrue(dt.Rows.Count > 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BloodProductSearch_Fail()
		{
		}

		[Test]
		public void GetBloodProductByGuid_Pass()
		{
			BOL.BloodProduct objTest = new BOL.BloodProduct();
			objTest.GetBloodProductByGuid(this._testBloodProduct.BloodProductGuid);

			Assert.IsTrue(this._testBloodProduct.BloodProductGuid == objTest.BloodProductGuid);
			Assert.IsTrue(objTest.IsNew == false);
		}

		[Test]
		public void GetBloodProductByGuid_Fail()
		{
			this._testBloodProduct.GetBloodProductByGuid(System.Guid.NewGuid());
			Assert.IsTrue(this._testBloodProduct.IsNew == true);
		}

		[Test]
		public void GetHcpcsAssignmentHistory_Pass()
		{
			DataTable dt = BOL.BloodProduct.GetHcpcsAssignmentHistory(this._testBloodProduct.BloodProductGuid);
			Assert.IsTrue(dt.Rows.Count > 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetHcpcsAssignmentHistory_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM BloodProduct WHERE BloodProductGuid = '" + this._testBloodProduct.BloodProductGuid.ToString() + "'");

			Assert.IsTrue((System.Guid)dr[TABLES.BloodProduct.BloodProductGuid] == this._testBloodProduct.BloodProductGuid);
			Assert.IsTrue(dr[TABLES.BloodProduct.ProductName].ToString() == this._testBloodProduct.ProductName);
			Assert.IsTrue(dr[TABLES.BloodProduct.ProductTypeCode].ToString() == this._testBloodProduct.ProductType.ProductTypeCode);
			Assert.IsTrue(dr[TABLES.BloodProduct.ProductCode].ToString() == this._testBloodProduct.ProductCode);
			Assert.IsTrue((short)dr[TABLES.BloodProduct.ProductVolume] == this._testBloodProduct.ProductVolume);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}

		[Test]
		public void RefreshProductShippers_Pass()
		{
			BOL.ProductShipper psBefore = (BOL.ProductShipper)this._testBloodProduct.ProductShippers[0];

			decimal tmpValue = psBefore.ReturnCreditPercentage - 1;
            if (tmpValue < 0)
                tmpValue = 0;
			System.Guid unitGuid = psBefore.LocalSupplierGuid;

			//Take the first shipper, then make a change to it
			UnitTestUtil.RunSQL("UPDATE ProductShipper SET ReturnCreditPercentage = " + tmpValue.ToString() + " WHERE ProductShipperGuid = '" + psBefore.ProductShipperGuid.ToString() + "'");

			//Now do the Refresh, and verify that the change took
			this._testBloodProduct.RefreshProductShippers();

			BOL.ProductShipper psAfter = this._testBloodProduct.ProductShippers.GetElementByLocalSupplierGuid(unitGuid);
			Assert.IsTrue(psAfter.ReturnCreditPercentage == tmpValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void RefreshProductShippers_Fail()
		{
		}

//		[Test]
//		//		public void GetBloodProductByCode_Pass()
//		{
//			BOL.BloodProduct objTest = new BOL.BloodProduct();
//			objTest.GetBloodProductByCode(this._testBloodProduct.ProductCode);
//	
//			Assertion.IsTrue(objTest.BloodProductGuid == this._testBloodProduct.BloodProductGuid);
//		}
//
//		[Test]
//		//		[ExpectedExceptionAttribute(typeof(BOL.NoRecordsFoundException))]
//		public void GetBloodProductByCode_Fail()
//		{
//			this._testBloodProduct.GetBloodProductByCode(string.Empty);
//		}

		[Test]
		public void GetBloodProductsByCode_Pass()
		{
			DataTable dt = BOL.BloodProduct.GetBloodProductsByCode(this._testBloodProduct.ProductCode);
			Assert.IsTrue(dt.Rows.Count > 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetBloodProductsByCode_Fail()
		{
		}

		[Test]
		public void GetActiveOrLocalBloodProductsByCode_Pass()
		{
			// Get 'L' (local) product code
			string query = "SELECT TOP 1 ProductCode FROM BloodProduct WHERE ProductCode LIKE '27%'";
			string productCode = Convert.ToString( UnitTestUtil.RunSQLGetDataRow(query)[TABLES.BloodProduct.ProductCode] );
			//
			DataTable dt = BOL.BloodProduct.GetActiveOrLocalBloodProductsByCode(productCode);
			Assert.IsTrue(dt.Rows.Count > 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetActiveOrLocalBloodProductsByCode_Fail()
		{
		}

		[Test]
		public void IsBloodProductActive_Pass()
		{
			System.Guid bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + this.TestDivisionCode + "'");
			bool tstValue = BOL.BloodProduct.IsBloodProductActive(bpGuid);
			Assert.IsTrue(tstValue);
		}

		[Test]
		public void IsBloodProductActive_Fail()
		{
			bool tstValue = BOL.BloodProduct.IsBloodProductActive(System.Guid.NewGuid());
			Assert.AreEqual(false, tstValue);
		}

		[Test]
		public void Save_Pass()
		{	
			this.RefreshData = true;

			string hcpcsCode = "P2029";
			System.Guid bpGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM BloodProduct WHERE BloodProductGuid NOT IN (SELECT DISTINCT BloodProductGuid FROM HcpcsAssignment WHERE DivisionCode = '" + this.TestDivisionCode + "')");
			BOL.BloodProduct tmpBP = new BOL.BloodProduct(bpGuid);

			tmpBP.HcpcsCode = hcpcsCode;		//This WAS a valid hccps code in 2005
			tmpBP.Save(Common.UpdateFunction.UnitTests);		//no return code? vbecs rocks!

			DataRow drTest1 = UnitTestUtil.RunSQLGetDataRow("SELECT HcpcsCode FROM HcpcsAssignment WHERE UnAssignedDate IS NULL AND BloodProductGuid = '" + bpGuid.ToString() + "' AND DivisionCode = '" + this.TestDivisionCode + "'");
			string tstValue1 = drTest1[TABLES.HcpcsAssignment.HcpcsCode].ToString().Trim();

			Assert.AreEqual(hcpcsCode, tstValue1, "New hcpcs code");

			hcpcsCode = "P2031";
			tmpBP = new BOL.BloodProduct(bpGuid);
			tmpBP.HcpcsCode = hcpcsCode;		//This also was valid in 2005
			tmpBP.Save(Common.UpdateFunction.UnitTests);	//no return code? vbecs rocks!

			DataRow drTest2 = UnitTestUtil.RunSQLGetDataRow("SELECT HcpcsCode FROM HcpcsAssignment WHERE UnAssignedDate IS NULL AND BloodProductGuid = '" + bpGuid.ToString() + "' AND DivisionCode = '" + this.TestDivisionCode + "'");
			string tstValue2 = drTest2[TABLES.HcpcsAssignment.HcpcsCode].ToString().Trim();

			Assert.AreEqual(hcpcsCode, tstValue2, "Update hcpcs code");
		}

		[Test]
		[ExpectedException(typeof(Common.RowVersionException))]
		public void Save_Fail()
		{
			this._testBloodProduct.ProductShippers[0].RowVersion = new byte[8];
			this._testBloodProduct.ProductShippers[0].IsDirty = true;

			this._testBloodProduct.Save(Common.UpdateFunction.UnitTests);
		}

		[Test]
		public void GetProductCost_Pass()
		{
			BOL.ProductShipper objTest = this._testBloodProduct.ProductShippers[0];

			decimal cost = this._testBloodProduct.GetProductCost(objTest.LocalSupplierGuid);

			Assert.IsTrue(cost == objTest.ProductCost);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(NoRecordsFoundException))]
		public void GetProductCost_Fail()
		{
			decimal cost = this._testBloodProduct.GetProductCost(System.Guid.NewGuid());
		}

		[Test]
		public void GetHcpcsCodes_Pass()
		{
			try
			{
				VistALinkUnitTestHarness.RequireConnection();
				DataTable dt = BOL.BloodProduct.GetHcpcsCodes();

				Assert.IsTrue(dt.Rows.Count > 0);
			}
			finally
			{
				VistALinkUnitTestHarness.ReleaseConnection();
			}
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetHcpcsCodes_Fail()
		{
		}


		[Test]
		public void HasB2Attribute_Pass()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 *, ProductCodeIndicator AS BarcodeType FROM BloodProduct WHERE ProductAttributeForm LIKE '%B2%'" );
			dr[TABLES.BloodProduct.ProductAttributeForm] = DBNull.Value;			//Property will double-check with DB for a value
			BOL.BloodProduct bp = new BOL.BloodProduct(dr);

			Assert.AreEqual(true, bp.HasB2Attribute, "InvalidBloodProduct" );

			Guid bloodProductGuid = bp.BloodProductGuid;
			bp = new BloodProduct(bloodProductGuid);
			Assert.AreEqual(true, bp.HasB2Attribute, "HasAttribute" );
		}


		[Test]
		public void HasB2Attribute_Fail()
		{
			System.Guid bloodProductGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM BloodProduct WHERE ProductAttributeForm NOT LIKE '%B2%'");
			BOL.BloodProduct bp = new BloodProduct(bloodProductGuid);
			
			Assert.AreEqual(false, bp.HasB2Attribute, "NoAttribute" );
		}
	}
}
#endif
